import { NextRequest, NextResponse } from "next/server";
import prisma from "@/lib/global/globalPrisma";

export async function PUT(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) return NextResponse.json({ message: "ID invalide." }, { status: 400 });

        const ct = req.headers.get("content-type") || "";
        if (!ct.includes("multipart/form-data")) {
            return NextResponse.json({ message: "Content-Type attendu: multipart/form-data." }, { status: 415 });
        }

        const form = await req.formData();
        const firstName = String(form.get("firstName") ?? "");
        const lastName = String(form.get("lastName") ?? "");
        const dobStr = String(form.get("dob") ?? "");
        const avatar = form.get("icon");

        const dob = new Date(dobStr);
        if (isNaN(dob.getTime())) {
            return NextResponse.json({ message: "Date invalide." }, { status: 400 });
        }

        let iconeUpdate: string | undefined;
        if (avatar && avatar instanceof File) {
            const buf = Buffer.from(await avatar.arrayBuffer());
            iconeUpdate = buf.toString("base64");
        }

        const updated = await prisma.user.update({
            where: { id },
            data: {
                firstName,
                lastName,
                dob,
                ...(iconeUpdate ? { icone: iconeUpdate } : {}),
            },
            select: { id: true, firstName: true, lastName: true, email: true, dob: true, icone: true },
        });

        return NextResponse.json(updated, { status: 200 });
    } catch (e: any) {
        console.error("PUT /api/users/[id] error:", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne." }, { status: 500 });
    }
}

export async function DELETE(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) {
            return NextResponse.json({ message: "ID invalide." }, { status: 400 });
        }

        await prisma.user.delete({
            where: { id },
        });

        return NextResponse.json({ message: "Compte supprimé avec succès." }, { status: 200 });
    } catch (e: any) {
        console.error("DELETE /api/user/[id] error:", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne." }, { status: 500 });
    }
}
